/*
   Copyright (C) 2013 Vincenzo Lo Cicero e Giovanni Ubaldi

   Authors:
      Vincenzo Lo Cicero <e-mail: vincenzo.locicero@libero.it>
      Giovanni Ubaldi <e-mail: ubaweb.giovanni@gmail.com>
    
   This is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
$ gcc -Wall -W -pedantic -O3 repl.c -o repl
$ ./repl /home/vincenzo/prova/sito
*/

#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>

#define HT_SIZE 8191
#define BLOCK_SIZE 4096

typedef enum tagStates
{
	S0,
	S1,
	S2,
	S3,
	S4,
	S5,
	S6,
	S7,
	S8,
	S9,
	S10,
	S11,
	S12,
	S13,
	S14,
	S15,
	S16,
	S17,
	S18,
	S19,
	S20,
	S21,
	S22,
	S23,
	S24
} States;

typedef struct tagFilesList
{
	char oldPathName[PATH_MAX + 1];
	char newPathName[PATH_MAX + 1];
	char oldFileName[PATH_MAX + 1];
	char newFileName[PATH_MAX + 1];
	char relativeName[PATH_MAX + 1];
	unsigned char isReferenced;
	unsigned char isHtml;
	struct tagFilesList *next;
} FilesList;

typedef struct tagHashTable
{
	FilesList *Key;
	struct tagHashTable *next;
} HashTable;

HashTable** htInit(HashTable **p);
HashTable* htNewNode(FilesList *Key);
int htFind(HashTable **pHashTable, FilesList *Key);
int htInsert(HashTable **pHashTable, FilesList *Key);
void htFreeList(HashTable* first);
void htFree(HashTable **p);

void printError(int numError);

HashTable** htInit(HashTable **p)
{
	int x;
	
	p = NULL;
	
	p = (HashTable**)malloc(sizeof(HashTable*) * HT_SIZE);
	if ( !p )
		return NULL;
		
	for ( x = 0; x < HT_SIZE; x++ )
		p[x] = NULL;
	
	return p;
}

HashTable* htNewNode(FilesList *Key)
{
	HashTable *n = NULL;

	n = (HashTable*)malloc(sizeof(HashTable));

	if( n == NULL )
		return NULL;

	n->Key = Key;
	
	n->next = NULL;

	return n;
}

int htFind(HashTable **pHashTable, FilesList *Key)
{
	int index = 0;
	HashTable *t = NULL;
	int a = 31415;
	char *s = Key->relativeName;
	
	int x = 0;

	for(; *s != '\0'; s++)
		index = (a*index + *s) % HT_SIZE;
	if ( index < 0 )
		index *= -1;

	t = pHashTable[index];
	while ( t != NULL )
	{
		if ( strcmp(t->Key->relativeName, Key->relativeName) == 0 )
		{
			t->Key->isReferenced = 1;
			return index;
		}
		x++;
					
		t = t->next;
	}

	return -1;
}

int htInsert(HashTable **pHashTable, FilesList *Key)
{
	int index = 0;
	HashTable *t = NULL;
	int a = 31415;
	char *s = Key->relativeName;

	for(; *s != '\0'; s++)
		index = (a*index + *s) % HT_SIZE;
	if ( index < 0 )
		index *= -1;

	t = pHashTable[index];
	if ( t == NULL )
	{
		pHashTable[index] = htNewNode(Key);
		return index;
	}

	while ( t != NULL )
	{
		if ( strcmp(t->Key->relativeName, Key->relativeName) == 0 )
		{
			return -1;
		}
		if ( t->next == NULL )
		{
			t->next = htNewNode(Key);
			return index;
		}
		t = t->next;
	}
	
	return index;
}

void htFreeList(HashTable* first)
{
	HashTable *n1 = first, *n2 = NULL;
	
	while ( n1 != NULL )
	{
		n2 = n1->next;		
		free(n1);
		n1 = NULL;
		n1 = n2;		
	}
}

void htFree(HashTable **p)
{
	int x;
	
	for( x = 0; x < HT_SIZE; x++ )
	{
		if ( p[x] )
			htFreeList(p[x]);
	}	
	
	free(p);
}

void freeFilesList(FilesList *first)
{
	FilesList *n1 = first, *n2;
	while ( n1 != NULL )
	{
		n2 = n1->next;
		free(n1);
		n1 = n2;
	}
}

FilesList* addFileToFilesList(FilesList *first, FilesList *newFile)
{
	FilesList *n;

	n = (FilesList*)malloc(sizeof(FilesList));

	if( n == NULL )
		return NULL;

	strcpy(n->oldPathName, newFile->oldPathName);
	strcpy(n->newPathName, newFile->newPathName);	
	strcpy(n->oldFileName, newFile->oldFileName);
	strcpy(n->newFileName, newFile->newFileName);
	strcpy(n->relativeName, newFile->relativeName);		
	n->isReferenced = newFile->isReferenced;
	n->isHtml = newFile->isHtml;
	n->next = NULL;
	
	if ( first != NULL )
		n->next = first;

	return n;
}

int createDirectory(char *dirName)
{
	DIR *dir = NULL;
	char pathName[PATH_MAX + 1];
	int len;
	int x = 0;
	int y = 0;

	strncpy(pathName, dirName, PATH_MAX);
	
	len = strlen(pathName);
	dir = opendir(pathName);
	if( dir == NULL )
	{
		if ( errno == ENOENT )
		{
			for ( x = len - 1; x >= 0; x-- )
			{
				if ( pathName[x] == '/' )
				{
					pathName[x] = '\0';
					dir = opendir(pathName);
					if ( dir != NULL )
					{
						closedir(dir);
						break;
					}
				}
			}
			
			strncpy(pathName, dirName, PATH_MAX);
			
			for ( y = x + 1; y <= len; y++ )
			{
				if ( pathName[y] == '/' || pathName[y] == '\0' )
				{
					pathName[y] = '\0';
					if ( mkdir(pathName, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) != 0 )
					{
						printf("Errore nella creazione della directory '%s'; %s.\n", pathName, strerror(errno));
						printError(errno);
						return 0;
					}
					else
					{
						pathName[y] = '/';
					}
				}
			}
			
			return 1;
		}
		else
		{		
			printf("Errore nell'apertura della directory '%s': %s\n", dirName, strerror(errno));
			return 0;
		}		
	}
		
	closedir(dir);
	
	return 1;
}

FilesList* getFilesRecursive(char *dirName, int lenOrig, HashTable **myTable, FilesList* myFilesList)
{
	FilesList myFile;		
	DIR *dir = NULL;
	struct dirent entry;
	struct dirent *entryPtr = NULL;
	/* int retval = 0; */
	char pathName[PATH_MAX + 1];
	char newPathName[PATH_MAX + 1];
	
	int i, len;
	char c;
	int index;
	char *szExt;	
	char *szTemp;	

	dir = opendir(dirName);
	if( dir == NULL )
	{
		printf("Errore nell'apertura della directory '%s': %s\n", dirName, strerror(errno));
		return myFilesList;
	}	
	
	/* retval = */ readdir_r(dir, &entry, &entryPtr);
	while( entryPtr != NULL )
	{
		struct stat entryInfo;
		
		if( ( strncmp(entry.d_name, ".", PATH_MAX) == 0 ) ||
		    ( strncmp(entry.d_name, "..", PATH_MAX) == 0 ) )
		{
			/* retval = */ readdir_r(dir, &entry, &entryPtr);
		    continue;
		}
		 		
		strncpy(pathName, dirName, PATH_MAX);
		strncat(pathName, "/", PATH_MAX);
		strncat(pathName, entry.d_name, PATH_MAX);
		
		szTemp = &pathName[lenOrig];
		strncpy(newPathName, pathName, PATH_MAX);
		newPathName[lenOrig] = '\0';
		strncat(newPathName, "_NewSite", PATH_MAX);
		strncat(newPathName, szTemp, PATH_MAX);										
		
		if( lstat(pathName, &entryInfo) == 0 )
		{						
			if( S_ISDIR(entryInfo.st_mode) )		/* directory */
			{								
				myFilesList = getFilesRecursive(pathName, lenOrig, myTable, myFilesList);
			}
			else if( S_ISREG(entryInfo.st_mode) )	/* regular file */
			{
				len = strlen(entry.d_name);
				
				strncpy(myFile.oldPathName, pathName, PATH_MAX);
				index = strlen(pathName) - len - 1;
				myFile.oldPathName[index] = '\0';
				
				strncpy(myFile.newPathName, newPathName, PATH_MAX);
				index = strlen(newPathName) - len - 1;
				myFile.newPathName[index] = '\0';
				
				strncpy(myFile.oldFileName, entry.d_name, PATH_MAX);
				index = 0;
				for ( i = 0; i < len; i++ )
				{
					c = myFile.oldFileName[i];
					if ( c >= 'A' && c <= 'Z' )
						myFile.newFileName[i] = myFile.oldFileName[i] + 32;
					else if ( c == ' ' )
						myFile.newFileName[i] = '_';
					else
						myFile.newFileName[i] = myFile.oldFileName[i];
						
					if ( myFile.newFileName[i] == '.' )
						index = i;
				}
				myFile.newFileName[i] = '\0';
				myFile.isReferenced = 0;
				myFile.isHtml = 0;
				
				myFile.relativeName[0] = '.';
				myFile.relativeName[1] = '\0';
				strncat(myFile.relativeName, &(myFile.oldPathName[lenOrig]), PATH_MAX);
				strncat(myFile.relativeName, "/", PATH_MAX);				
				strncat(myFile.relativeName, myFile.newFileName, PATH_MAX);
								
				if ( index > 0 )
				{
					szExt = &myFile.newFileName[index];
					if ( (strcmp(".htm", szExt) == 0) || (strcmp(".html", szExt) == 0 ) )
						myFile.isHtml = 1;
				}
				
				myFilesList = addFileToFilesList(myFilesList, &myFile);
				if ( !myFilesList )
					return NULL;
									
				htInsert(myTable, myFilesList);				
			}			
		}
		else
		{
			printf("Errore lstat %s: %s\n", pathName, strerror(errno));
		}

		/* retval = */ readdir_r(dir, &entry, &entryPtr);
	}
		
	closedir(dir);	
	
	return myFilesList;
}

void printError(int numError)
{
	switch ( numError )
	{
		case EACCES:
			printf("Search permission is denied on a component of the path prefix, or write permission is denied on the parent directory of the directory to be created.\n");
			break;
		case EEXIST:
			printf("The named file exists.\n");
			break;
		case ELOOP:
			printf("A loop exists in symbolic links encountered during resolution of the path argument.\n");
			break;
		case EMLINK:
			printf("The link count of the parent directory would exceed {LINK_MAX}.\n");		
			break;
		case ENAMETOOLONG:
			printf("The length of the path argument exceeds {PATH_MAX} or a pathname component is longer than {NAME_MAX}.\n");
			break;
		case ENOENT:
			printf("A component of the path prefix specified by path does not name an existing directory or path is an empty string.\n");
			break;
		case ENOSPC:
			printf("The file system does not contain enough space to hold the contents of the new directory or to extend the parent directory of the new directory.\n");
			break;
		case ENOTDIR:
			printf("A component of the path prefix is not a directory.\n");		
			break;
		case EROFS:
			printf("The parent directory resides on a read-only file system.\n");		
			break;
		default:
			printf("Unknown error\n");
			break;						
	}
}

int readFiles(const char *szPath, FilesList* myFilesList, HashTable **pHashTable, int lenOrig)
{
	int retValue = 1;
	FILE *fp = NULL;
	FILE *fpNew = NULL;	
	FILE *fpNonExistentFiles = NULL;		
	unsigned char *myBlock = NULL;
	unsigned char *myBlockNew = NULL;	
	FilesList *n;
	char szFileName[PATH_MAX];
	char szNewFileName[PATH_MAX];
	char szNonExistentFileName[PATH_MAX];
	
	int row = 1;
	int blockLen = 0;
	int index1 = 0;	
	int index2 = 0;
	
	States state = S0;
	char c;
	char lexeme[1024];
	int k = 0;
	int x, y, z;
	int w;
	int lenLexeme;
	
	char szLine[1024];
	char szTemp[PATH_MAX];	
	
	int bSlash = 0;
	int bColon = 0;
	int bPoint = 0;
	int bChiocciola = 0;
				
	myBlock = (unsigned char *)malloc(sizeof(unsigned char) * BLOCK_SIZE);
	if ( !myBlock )
	{
		printf("Memoria insufficiente.\n");
		retValue = 0;
		goto uscita;
	}

	myBlockNew = (unsigned char *)malloc(sizeof(unsigned char) * BLOCK_SIZE);
	if ( !myBlockNew )
	{
		printf("Memoria insufficiente.\n");
		retValue = 0;
		goto uscita;		
	}
	
	strncpy(szNonExistentFileName, szPath, PATH_MAX);
	strncat(szNonExistentFileName, "_NewSite/DeadLinks.txt", PATH_MAX);		
	fpNonExistentFiles = fopen(szNonExistentFileName, "wb");
	if ( fpNonExistentFiles == NULL )
	{
		printf("Errore nell'apertura del file '%s'.\n", szNonExistentFileName);
		retValue = 0;
		goto uscita;
	}
	
	
	n = myFilesList;
	while( n != NULL )
	{
		strncpy(szFileName, n->oldPathName, PATH_MAX);
		strncat(szFileName, "/", PATH_MAX);
		strncat(szFileName, n->oldFileName, PATH_MAX);
		
		strncpy(szNewFileName, n->newPathName, PATH_MAX);
		strncat(szNewFileName, "/", PATH_MAX);
		strncat(szNewFileName, n->newFileName, PATH_MAX);		
		
		if ( fp != NULL )
		{
			fclose(fp);
			fp = NULL;			
		}
		
		if ( fpNew != NULL )
		{
			fclose(fpNew);		
			fpNew = NULL;
		}		
		
		fp = fopen(szFileName, "rb");
		if ( fp == NULL )
		{
			printf("Errore nell'apertura del file '%s'.\n", szFileName);
			retValue = 0;
			goto uscita;
		}
		
		fpNew = fopen(szNewFileName, "wb");
		if ( fpNew == NULL )
		{
			/* sprintf(szLine, "mkdir -p %s", n->newPathName); */
			/* system(szLine); */
			
			if ( !createDirectory(n->newPathName) )
				goto uscita;
						
			fpNew = fopen(szNewFileName, "wb");
			if ( fpNew == NULL )
			{
				printf("Errore nella creazione del file '%s'.\n", szNewFileName);
				n = n->next;
				continue;
			}
		}
		
		row = 1;
		blockLen = 0;
		index1 = 0;	
		index2 = 0;
		k = 0;	
		
		while ( (blockLen = fread(myBlock, 1, BLOCK_SIZE, fp)) )
		{
			if ( !n->isHtml )
			{
				fwrite(myBlock, 1, blockLen, fpNew);
				continue;
			}
								
			while ( index1 < blockLen )
			{
				c = myBlock[index1];
				
				if ( c == '\n' )
					row++;
				
				switch ( state )
				{
					case S0:
						if ( c == '<' )
						{
							myBlockNew[index2++] = c;
							state = S1;
						}
						else
						{
							myBlockNew[index2++] = c;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
					case S1:
						if ( c >= 'a' && c <= 'z' )
						{
							myBlockNew[index2++] = c;
							lexeme[k++] = c;
						}
						else if ( c >= 'A' && c <= 'Z' )
						{
							myBlockNew[index2++] = c;
							c += 32;
							lexeme[k++] = c;
						}
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;
							lexeme[k] = '\0';
							k = 0;
							
							state = S2;
															
							/*						
							if ( strcmp(lexeme, "a") == 0 )
								state = S2;
							else if ( strcmp(lexeme, "img") == 0 )
								state = S2;
							else if ( strcmp(lexeme, "frame") == 0 )
								state = S2;
							else
								state = S0;
							*/
						}
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S0;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S2:
						if ( c == ' ' )
						{
							myBlockNew[index2++] = c;							
						}
						else if ( c >= 'a' && c <= 'z' )
						{
							myBlockNew[index2++] = c;
							lexeme[k++] = c;
							state = S3;
						}
						else if ( c >= 'A' && c <= 'Z' )
						{
							myBlockNew[index2++] = c;
							c += 32;
							lexeme[k++] = c;
							state = S3;
						}						
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S0;							
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S3:
						if ( c >= 'a' && c <= 'z' )
						{
							myBlockNew[index2++] = c;
							lexeme[k++] = c;
						}
						else if ( c >= 'A' && c <= 'Z' )
						{
							myBlockNew[index2++] = c;
							c += 32;
							lexeme[k++] = c;
						}
						else if ( c == '=' || c == ' ' )
						{
							myBlockNew[index2++] = c;
							lexeme[k] = '\0';
							k = 0;
																													
							if ( strcmp(lexeme, "href") == 0 )
							{
								if ( c == '=' )
									state = S5;
								else
									state = S4;
							}
							else if ( strcmp(lexeme, "src") == 0 )
							{
								if ( c == '=' )
									state = S5;
								else
									state = S4;
							}
							else if ( strcmp(lexeme, "style") == 0 )
							{
								state = S14;
							}							
							else if ( strcmp(lexeme, "value") == 0 )
							{
								if ( c == '=' )
									state = S5;
								else
									state = S4;
							}
							else if ( strcmp(lexeme, "background") == 0 )
							{
								if ( c == '=' )
									state = S5;
								else
									state = S4;
							}							
							else
							{
								state = S9;								
							}
						}
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S0;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S4:
						if ( c == '=' )
						{
							myBlockNew[index2++] = c;
							state = S5;
						}					
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;							
						}
						else if ( c >= 'a' && c <= 'z' )
						{
							myBlockNew[index2++] = c;
							k = 0;
							lexeme[k++] = c;
							state = S3;
						}
						else if ( c >= 'A' && c <= 'Z' )
						{
							myBlockNew[index2++] = c;
							c += 32;
							k = 0;
							lexeme[k++] = c;
							state = S3;
						}												
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S3;
							/* state = S0; */
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S5:
						if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S6;
						}					
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;
						}					
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S3;
							/* state = S0; */
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
											
					case S6:
						if ( c == '"' || c == '#' )
						{
							FilesList Key;
							
							szLine[k] = '\0';
							lexeme[k] = '\0';
							k = 0;
							myBlockNew[index2++] = c;
							state = S3;
														
							lenLexeme = strlen(lexeme) - 1;
							for ( y = lenLexeme; y >= 0; y-- )
							{
								if ( lexeme[y] == '/' || lexeme[y] == '\\')
								{
									lexeme[y] = '/';
									break;
								}
							}
							y++;
							
							if ( lenLexeme > 2 && szLine[0] == '.' && szLine[1] == '/' )
							{
								Key.relativeName[0] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1] = '/';
								Key.relativeName[y] = '\0';								
							}
							else if ( lenLexeme > 3 && szLine[0] == '.' && szLine[1] == '.' && szLine[2] == '/' )
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
															
								w = 0;
								while ( w < lenOrig )
								{
									szTemp[w] = n->oldPathName[w];
									w++;
								}
								szTemp[w] = '\0';
								
								if ( szTemp[lenOrig] != '\0' )
								{
									strncat(Key.relativeName, &szTemp[lenOrig + 1], PATH_MAX);
									strncat(Key.relativeName, "/", PATH_MAX);
								}
								strncat(Key.relativeName, &szLine[3], PATH_MAX);
								
								w = strlen(lexeme);
								z = strlen(Key.relativeName);
								while ( w > 0 && z > 0 )
								{
									if ( lexeme[w] == '/' )
										break;
									Key.relativeName[z] = lexeme[w];
									--w;
									--z;
								}
																
								w = strlen(Key.relativeName) - 1;
								while ( w > 0 )
								{										
									if ( Key.relativeName[w] == '/' || Key.relativeName[w] == '\\' )
									{
										Key.relativeName[w+1] = '\0';
										break;
									}
									--w;
								}								
							}
							else
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1 + 3] = '\0';																
							}
							
							for ( z = 0; ; z++ )
							{
								Key.newFileName[z] = lexeme[y++];
								if ( Key.newFileName[z] == '\0' )
									break;
							}
							
							strncat(Key.relativeName, Key.newFileName, PATH_MAX);
														
							w = strlen(Key.relativeName) - 1;
							while ( w > 0 )
							{										
								if ( Key.relativeName[w] == '\\' )
									Key.relativeName[w] = '/';
								--w;
							}								
																																			
							if ( bSlash )
							{
								lenLexeme = strlen(szLine);
								x = (index2 - 1) - lenLexeme;
								for ( z = lenLexeme - 1; z > 0; z-- )
								{
									if ( szLine[z] == '/' || szLine[z] == '\\' )
									{
										szLine[z] = '\0';
										break;
									}
								} 	
								z = 0;
								if ( x >= 0 )
								{
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}
								else
								{
									fseek(fpNew, x, SEEK_CUR);
									while ( x < 0 )
									{
										fwrite(&szLine[z], 1, 1, fpNew);
										x++;
										z++;
									}
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}								
							}
														
							if ( htFind(pHashTable, &Key) == -1 )
							{		
								if ( bColon == 0 && bChiocciola == 0 && bPoint == 1 && !isdigit(lexeme[0])  )
								{																		
									sprintf(szLine, "%s/%s\t%d\t%s\n", n->newPathName, n->newFileName, row, lexeme);
									fwrite(szLine, 1, strlen(szLine), fpNonExistentFiles);
								}
							}
														
							bSlash = 0;
							bColon = 0;
							bPoint = 0;
							bChiocciola = 0;
						}
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = '_';
							szLine[k] = '_';
							lexeme[k++] = '_';						
						}
						else if ( c == '%' )
						{
							state = S7;
							break;
						}
						else
						{
							if ( c != '\n' )
								szLine[k] = c;
							
							if ( c >= 'A' && c <= 'Z' )
								c += 32;
								
							if ( c != '\n' )
							{
								if ( c == '\\' )
									c = '/';
								myBlockNew[index2++] = c;
								lexeme[k++] = c;
							}

							if ( c == '/' )
								bSlash = 1;
							
							if ( c == ':' )
								bColon = 1;
							
							if ( c == '.' )
								bPoint = 1;								
							
							if ( c == '@' )
								bChiocciola = 1;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S7:
						if ( c == '2' )
						{
							state = S8;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = c;
							k = 0;
							state = S3;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S8:
						if ( c == '0' )
						{
							szLine[k] = '_';
							lexeme[k++] = '_';
							myBlockNew[index2++] = '_';
							state = S6;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = '2';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
														
							myBlockNew[index2++] = c;
							k = 0;
							state = S3;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S9:
						if ( c == ' ' )
						{
							myBlockNew[index2++] = c;
						}
						else if ( c == '=' )
						{
							myBlockNew[index2++] = c;
						}
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S10;
						}		
						else if ( c >= 'a' && c <= 'z' )
						{
							myBlockNew[index2++] = c;
							szLine[k] = c;
							lexeme[k++] = c;
							state = S2;
						}
						else if ( c >= 'A' && c <= 'Z' )
						{
							myBlockNew[index2++] = c;
							szLine[k] = c;
							c += 32;
							lexeme[k++] = c;
							state = S2;
						}
						else
						{
							myBlockNew[index2++] = c;
							state = S0;
						}												
					
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
											
						break;
						
					case S10:
						if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S2;
						}
						else if ( c == '\'' )
						{
							myBlockNew[index2++] = c;
							state = S11;
						}												
						else
						{
							myBlockNew[index2++] = c;
						}												
					
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
											
						break;						
						
					case S11:
						if ( c == '\'' )
						{
							FilesList Key;
							
							szLine[k] = '\0';
														
							lexeme[k] = '\0';
							k = 0;
							myBlockNew[index2++] = c;
							state = S10;
														
							lenLexeme = strlen(lexeme) - 1;
							for ( y = lenLexeme; y >= 0; y-- )
							{
								if ( lexeme[y] == '/' || lexeme[y] == '\\')
								{
									lexeme[y] = '/';
									break;
								}
							}
							y++;
							
							if ( lenLexeme > 2 && szLine[0] == '.' && szLine[1] == '/' )
							{
								Key.relativeName[0] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1] = '/';
								Key.relativeName[y] = '\0';								
							}
							else if ( lenLexeme > 3 && szLine[0] == '.' && szLine[1] == '.' && szLine[2] == '/' )
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
															
								w = 0;
								while ( w < lenOrig )
								{
									szTemp[w] = n->oldPathName[w];
									w++;
								}
								szTemp[w] = '\0';
								
								if ( szTemp[lenOrig] != '\0' )
								{
									strncat(Key.relativeName, &szTemp[lenOrig + 1], PATH_MAX);
									strncat(Key.relativeName, "/", PATH_MAX);
								}
								strncat(Key.relativeName, &szLine[3], PATH_MAX);
								
								w = strlen(lexeme);
								z = strlen(Key.relativeName);
								while ( w > 0 && z > 0 )
								{
									if ( lexeme[w] == '/' )
										break;
									Key.relativeName[z] = lexeme[w];
									--w;
									--z;
								}
																
								w = strlen(Key.relativeName) - 1;
								while ( w > 0 )
								{										
									if ( Key.relativeName[w] == '/' || Key.relativeName[w] == '\\' )
									{
										Key.relativeName[w+1] = '\0';
										break;
									}
									--w;
								}																
							}
							else
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1 + 3] = '\0';
							}
														
							for ( z = 0; ; z++ )
							{
								Key.newFileName[z] = lexeme[y++];
								if ( Key.newFileName[z] == '\0' )
									break;
							}
							
							strncat(Key.relativeName, Key.newFileName, PATH_MAX);
							
							w = strlen(Key.relativeName) - 1;
							while ( w > 0 )
							{										
								if ( Key.relativeName[w] == '\\' )
									Key.relativeName[w] = '/';
								--w;
							}															
														
							if ( bSlash )
							{
								lenLexeme = strlen(szLine);
								x = (index2 - 1) - lenLexeme;
								for ( z = lenLexeme - 1; z > 0; z-- )
								{
									if ( szLine[z] == '/' || szLine[z] == '\\' )
									{
										szLine[z] = '\0';
										break;
									}
								} 																
								z = 0;
								if ( x >= 0 )
								{
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}
								else
								{
									fseek(fpNew, x, SEEK_CUR);
									while ( x < 0 )
									{
										fwrite(&szLine[z], 1, 1, fpNew);
										x++;
										z++;
									}
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}								
							}							
							
							if ( htFind(pHashTable, &Key) == -1 )
							{
								if ( bColon == 0 && bChiocciola == 0 && bPoint == 1 && !isdigit(lexeme[0])  )
								{
									sprintf(szLine, "%s/%s\t%d\t%s\n", n->newPathName, n->newFileName, row, lexeme);
									fwrite(szLine, 1, strlen(szLine), fpNonExistentFiles);
								}
								else
								{									
									x = (index2 - 1) - (strlen(szLine));
									z = 0;
									if ( x >= 0 )
									{
										while ( szLine[z] != '\0' )
										{
											myBlockNew[x] = szLine[z];
											x++;
											z++;
										}
									}
									else
									{
										fseek(fpNew, x, SEEK_CUR);
										while ( x < 0 )
										{
											fwrite(&szLine[z], 1, 1, fpNew);
											x++;
											z++;
										}
										while ( szLine[z] != '\0' )
										{
											myBlockNew[x] = szLine[z];
											x++;
											z++;
										}
									}
								}
							}
														
							bSlash = 0;
							bColon = 0;
							bPoint = 0;
							bChiocciola = 0;							
						}																	
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S2;
						}
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = '_';
							szLine[k] = '_';
							lexeme[k++] = '_';
						}
						else if ( c == '%' )
						{
							state = S12;
							break;
						}
						else
						{
							if ( c != '\n' )
							{
								szLine[k] = c;
							}
							
							if ( c >= 'A' && c <= 'Z' )
								c += 32;
								
							if ( c != '\n' )
							{
								if ( c == '\\' )
									c = '/';								
								myBlockNew[index2++] = c;
								lexeme[k++] = c;
							}
							
							if ( c == '/' )
								bSlash = 1;
							
							if ( c == ':' )
								bColon = 1;
							
							if ( c == '.' )
								bPoint = 1;								
							
							if ( c == '@' )
								bChiocciola = 1;
						}
					
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
											
						break;
						
					case S12:
						if ( c == '2' )
						{
							state = S13;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = c;
							k = 0;
							state = S10;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S13:
						if ( c == '0' )
						{
							szLine[k] = '_';
							lexeme[k++] = '_';
							myBlockNew[index2++] = '_';
							state = S11;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = '2';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
														
							myBlockNew[index2++] = c;
							k = 0;
							state = S10;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S14:
						k = 0;
						bColon = 0;
						bPoint = 0;
						bChiocciola = 0;
						
						if ( c == '=' )
						{
							myBlockNew[index2++] = c;
							state = S16;
						}					
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S16;
						}											
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;							
						}
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S3;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S15:
						if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S3;
						}					
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;
						}					
						else
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S16;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S16:
						if ( c != ':' )
						{
							myBlockNew[index2++] = c;
						}
						else
						{
							myBlockNew[index2++] = c;
							state = S17;
						}											
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S17:
						if ( c == ' ' )
						{
							myBlockNew[index2++] = c;							
						}
						else if ( c == ';' )
						{
							myBlockNew[index2++] = c;
							state = S15;
						}						
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;							
							state = S3;
						}
						else if ( c == 'u' )
						{
							myBlockNew[index2++] = c;
							state = S18;
						}
						else if ( c == 'U' )
						{
							myBlockNew[index2++] = c;
							state = S18;
						}
						else
						{
							myBlockNew[index2++] = c;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;						
						
					case S18:
						if ( c == ';' )
						{
							myBlockNew[index2++] = c;
							state = S15;
						}						
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;							
							state = S3;
						}					
						else if ( c == 'r' )
						{
							myBlockNew[index2++] = c;
							state = S19;
						}
						else if ( c == 'R' )
						{
							myBlockNew[index2++] = c;
							state = S19;
						}												
						else
						{
							myBlockNew[index2++] = c;
							if ( c == ';' )
								state = S14;
							else if ( c == '"' )
								state = S3;															
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S19:
						if ( c == ';' )
						{
							myBlockNew[index2++] = c;
							state = S15;
						}						
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;							
							state = S3;
						}										
						else if ( c == 'l' )
						{
							myBlockNew[index2++] = c;
							state = S20;
						}
						else if ( c == 'L' )
						{
							myBlockNew[index2++] = c;
							state = S20;
						}												
						else
						{
							myBlockNew[index2++] = c;
							if ( c == ';' )
								state = S14;
							else if ( c == '"' )
								state = S3;																						
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S20:
						if ( c == ';' )
						{
							myBlockNew[index2++] = c;
							state = S15;
						}						
						else if ( c == '"' )
						{
							myBlockNew[index2++] = c;							
							state = S3;
						}										
						else if ( c == '(' )
						{
							myBlockNew[index2++] = c;
							state = S22;
						}
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = c;							
						}
						else
						{
							myBlockNew[index2++] = c;
							if ( c == ';' )
								state = S14;
							else if ( c == '"' )
								state = S3;								
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S21:
						if ( c == '"' )
						{
							myBlockNew[index2++] = c;
							state = S3;
						}
						else if ( c == ';' )
						{
							myBlockNew[index2++] = c;
							state = S17;
						}						
						else if ( c == '>' )
						{
							myBlockNew[index2++] = c;
							k = 0;
							state = S0;
						}												
						else
						{
							myBlockNew[index2++] = c;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S22:
						if ( c == ')' )
						{
							FilesList Key;
							
							szLine[k] = '\0';
							lexeme[k] = '\0';
							k = 0;
							myBlockNew[index2++] = c;
							state = S21;
																				
							lenLexeme = strlen(lexeme) - 1;
							for ( y = lenLexeme; y >= 0; y-- )
							{
								if ( lexeme[y] == '/' || lexeme[y] == '\\')
								{
									lexeme[y] = '/';
									break;
								}
							}
							y++;
							
							if ( lenLexeme > 2 && szLine[0] == '.' && szLine[1] == '/' )
							{
								Key.relativeName[0] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1] = '/';
								Key.relativeName[y] = '\0';								
							}
							else if ( lenLexeme > 3 && szLine[0] == '.' && szLine[1] == '.' && szLine[2] == '/' )
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
															
								w = 0;
								while ( w < lenOrig )
								{
									szTemp[w] = n->oldPathName[w];
									w++;
								}
								szTemp[w] = '\0';
								
								if ( szTemp[lenOrig] != '\0' )
								{
									strncat(Key.relativeName, &szTemp[lenOrig + 1], PATH_MAX);
									strncat(Key.relativeName, "/", PATH_MAX);
								}
								strncat(Key.relativeName, &szLine[3], PATH_MAX);
								
								w = strlen(lexeme);
								z = strlen(Key.relativeName);
								while ( w > 0 && z > 0 )
								{
									if ( lexeme[w] == '/' )
										break;
									Key.relativeName[z] = lexeme[w];
									--w;
									--z;
								}
																
								w = strlen(Key.relativeName) - 1;
								while ( w > 0 )
								{										
									if ( Key.relativeName[w] == '/' || Key.relativeName[w] == '\\' )
									{
										Key.relativeName[w+1] = '\0';
										break;
									}
									--w;
								}																
							}
							else
							{
								Key.relativeName[0] = '.';
								Key.relativeName[1] = '/';
								Key.relativeName[2] = '\0';
								strncat(Key.relativeName, szLine, PATH_MAX);
								Key.relativeName[y - 1 + 3] = '\0';
							}
														
							for ( z = 0; ; z++ )
							{
								Key.newFileName[z] = lexeme[y++];
								if ( Key.newFileName[z] == '\0' )
									break;
							}
							
							strncat(Key.relativeName, Key.newFileName, PATH_MAX);
							
							w = strlen(Key.relativeName) - 1;
							while ( w > 0 )
							{										
								if ( Key.relativeName[w] == '\\' )
									Key.relativeName[w] = '/';
								--w;
							}															
														
							if ( bSlash )
							{
								lenLexeme = strlen(szLine);
								x = (index2 - 1) - lenLexeme;
								for ( z = lenLexeme - 1; z > 0; z-- )
								{
									if ( szLine[z] == '/' || szLine[z] == '\\' )
									{
										szLine[z] = '\0';
										break;
									}
								}								
								z = 0;
								if ( x >= 0 )
								{
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}
								else
								{
									fseek(fpNew, x, SEEK_CUR);
									while ( x < 0 )
									{
										fwrite(&szLine[z], 1, 1, fpNew);
										x++;
										z++;
									}
									while ( szLine[z] != '\0' )
									{
										myBlockNew[x] = szLine[z];
										x++;
										z++;
									}
								}								
							}
														
							if ( htFind(pHashTable, &Key) == -1 )
							{
								if ( bColon == 0 && bChiocciola == 0 && bPoint == 1 && !isdigit(lexeme[0])  )
								{
									sprintf(szLine, "%s/%s\t%d\t%s\n", n->newPathName, n->newFileName, row, lexeme);
									fwrite(szLine, 1, strlen(szLine), fpNonExistentFiles);
								}
							}
														
							bSlash = 0;
							bColon = 0;
							bPoint = 0;
							bChiocciola = 0;
						}
						else if ( c == ' ' )
						{
							myBlockNew[index2++] = '_';
							szLine[k] = '_';
							lexeme[k++] = '_';						
						}
						else if ( c == '%' )
						{
							state = S23;
							break;
						}
						else
						{
							if ( c != '\n' )
								szLine[k] = c;
							
							if ( c >= 'A' && c <= 'Z' )
								c += 32;
								
							if ( c != '\n' )
							{
								if ( c == '\\' )
									c = '/';								
								myBlockNew[index2++] = c;
								lexeme[k++] = c;
							}
							
							if ( c == '/' )
								bSlash = 1;
							
							if ( c == ':' )
								bColon = 1;
							
							if ( c == '.' )
								bPoint = 1;								
							
							if ( c == '@' )
								bChiocciola = 1;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S23:
						if ( c == '2' )
						{
							state = S24;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = c;
							k = 0;
							state = S21;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
						
					case S24:
						if ( c == '0' )
						{
							szLine[k] = '_';
							lexeme[k++] = '_';
							myBlockNew[index2++] = '_';
							state = S22;
						}
						else
						{
							myBlockNew[index2++] = '%';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
							
							myBlockNew[index2++] = '2';
							if ( index2 >= BLOCK_SIZE )
							{
								fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
								index2 = 0;
							}
														
							myBlockNew[index2++] = c;
							k = 0;
							state = S21;
						}
						
						if ( index2 >= BLOCK_SIZE )
						{
							fwrite(myBlockNew, 1, BLOCK_SIZE, fpNew);
							index2 = 0;
						}
						
						break;
				}

				index1++;
			}
			if ( index2 > 0 && index2 < BLOCK_SIZE )
				fwrite(myBlockNew, 1, index2, fpNew);
			index1 = 0;
			index2 = 0;
		}		
								
		n = n->next;		
	}	
	
	uscita:
	
	if ( myBlock != NULL )
		free(myBlock);
		
	if ( myBlockNew != NULL )
		free(myBlockNew);
	
	if ( fp != NULL )
		fclose(fp);
		
	if ( fpNew != NULL )
		fclose(fpNew);
		
	if ( fpNonExistentFiles != NULL )
		fclose(fpNonExistentFiles);
	
	return retValue;
}

int main(int argc, char **argv)
{
	FilesList* myFilesList = NULL;	
	HashTable **myTable = NULL;
	int len;
	
	char szPathNewDir[PATH_MAX];
	
	char szFileName[PATH_MAX];
	FILE *fp = NULL;
	FilesList *n;	
	char *cNewLine = "\n";
				
	if(argc != 2)
	{
		printf("uso: %s <dirpath>\n", argv[0]);
		return -1;
	}
	
	myTable = htInit(myTable);
	if ( !myTable )
	{
		printf("Errore: impossibile inizializzare la tabella.\n");
		return -1;
	}
	
	len = strlen(argv[1]);
	
	myFilesList = getFilesRecursive(argv[1], len, myTable, myFilesList);	
	if ( !myFilesList )
	{
		printf("Errore: impossibile ottenere la lista dei file.\n");
		htFree(myTable);
		return -1;
	}
	
	strncpy(szPathNewDir, argv[1], PATH_MAX);	
	strncat(szPathNewDir, "_NewSite", PATH_MAX);
	
	mkdir(szPathNewDir, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
		
	readFiles(argv[1], myFilesList, myTable, len);	
	
	strncpy(szFileName, argv[1], PATH_MAX);
	strncat(szFileName, "_NewSite/UnreferencedFiles.txt", PATH_MAX);	
	fp = fopen(szFileName, "wb");
	if ( fp == NULL )
	{
		printf("Errore nell'apertura del file '%s'.\n", szFileName);
		goto uscita;
	}

	n = myFilesList;
	while( n != NULL )
	{					
		if ( n->isReferenced == 0 )
		{
			strcpy(szFileName, n->newPathName);
			strcat(szFileName, "/");
			strcat(szFileName, n->newFileName);
			fwrite(szFileName, 1, strlen(szFileName), fp);
			fwrite(cNewLine, 1, strlen(cNewLine), fp);			
		}
		
		n = n->next;
	}		
	
	uscita:
	
	if ( fp != NULL )
		fclose(fp);
	
	freeFilesList(myFilesList);
	
	htFree(myTable);
		
	return 0;
}
